;; --------------------------------------------------------------------------
;; Particle Urban Pigeon Inspired (PUPI) Algorithm for
;; Unconstrained Numerical Optimization.
;;
;; A model by Sergio Rojas-Galeano and Martha Garzon
;; v1.16 Copyright (c) July 2020 The authors
;; Correspondance email: srojas@udistrital.edu.co
;; Universidad Distrital Francisco Jose de Caldas, Bogota, Colombia
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License (GPLv3)
;; (see license at: https://www.gnu.org/licenses/gpl-3.0.txt)
;;
;; The model is made publicly available in the hope that it will be useful
;; to modelers, but WITHOUT ANY WARRANTY whatsoever (see license for details).
;; --------------------------------------------------------------------------

globals[
  ;; PUPI globals
  pupi-leader         ; best pigeon in current iteration
  pupi-leader-fitness ; highest value found by PUPI
  pupi-best-patch     ; best patch found by PUPI
  pupi-runtime        ; total algorithm runtime (ms)
  pupi-cohesion       ; flock cohesion
  pupi-best-tick      ; tick where optimum was found

  ;; Problem variables
  true-best-patch   ; patch with the true best value
]

patches-own[
  x     ; simulated pxcor, depending on the bounds range of vars
  y     ; simulated pycor, depending on the bounds range of vars
  value ; each patch has a value depending on cost_function and its coordinates
        ; the goal of PUPI algorithm is to find the patch with the best fitness value within the search space
]

;; PUPI breeds
breed [walkers walker]
breed [followers follower]


;; Create the fitness landscape depending on optimisation problem
to setup-search-landscape
  clear-all

  ;; make a landscape with hills and valleys according to chosen cost function
  ask patches [
    set x pxcor * (xy-bounds / max-pxcor)
    set y pycor * (xy-bounds / max-pycor)

    set value (ifelse-value
      landscape = "Sphere" [
        x ^ 2 + y ^ 2
      ]
    	landscape = "Sphere-offset" [
        (x - 50 * (xy-bounds / max-pxcor) ) ^ 2  + (y + 50 * (xy-bounds / max-pxcor) ) ^ 2
      ]
      landscape = "Rastrigin" [ ; note that degrees, not radians, are needed for cos function
        20 + ((x ^ 2) -  10 * cos ( (180 / pi) * (2 * pi) * x )) + ((y ^ 2) -  10 * cos ( (180 / pi) * (2 * pi) * y ))
    	]
      landscape = "Rosenbrock" [
        100 * (y - (x ^ 2))^ 2 + (1 - x)^ 2
      ]
      landscape = "Himmelblau" [
        ((x ^ 2) + y - 11) ^ 2 + (x + (y ^ 2) - 7)^ 2
      ]
      landscape = "Eggholder" [ ; note that degrees, not radians, are needed for sin function
        ( (- x) * sin ( (180 / pi) * sqrt (abs (x - (y + 47))))) - (y + 47) * sin ( (180 / pi) * sqrt (abs ((x / 2) + (y + 47))))
      ]
      [ random-normal 0 500  ] ; the last case is a random landscape
    )
  ]

  if landscape = "Random" [
    ask min-one-of patches [value][ set value value - 500 ]
    repeat 10 [ diffuse value 1 ]
  ]

  ;; find the true best value
  ask min-one-of patches [value][ set true-best-patch self ]

	;; scale patches color within values limits
  let min-val min [value] of patches
  let max-val max [value] of patches
	ask patches [ set pcolor scale-color yellow value min-val log abs max-val 1.05 ]
end

to setup
  setup-search-landscape

  ;; create PUPI breeds of pigeons and place them randomly in the world
  create-walkers pop-size * walkers-rate [
    setxy random-xcor random-ycor  ; set walker pigeons starting position
    set color green                ; assing walker color
    set size 8										 ; make pigeons slightly bigger
  ]
  create-followers pop-size - count walkers [
    setxy random-xcor random-ycor  ; set follower pigeons starting position
    set color blue                 ; assing walker color
    set size 8										 ; make pigeons slightly bigger
  ]
  ;; initialise pupi best patch randomly
  set pupi-best-patch patch random-xcor random-ycor

  reset-ticks
end

to go
  reset-timer
;    ifelse ticks mod 1000 > 800 [
  ifelse ticks mod 500 > 400 [
    ;; PUPI wild search (starvation) moves
    ask (turtle-set followers walkers) [ walk-move ]
  ][
    ;; PUPI normal search moves
    find-leader
    ask followers [ follow-move ]
    ask walkers [ walk-move ]
    ask pupi-leader [ set color red ]
  ]
  set pupi-runtime pupi-runtime + timer
  if cohesion? [ set pupi-cohesion sum [distance pupi-leader] of followers ]

  update-spotlight
  tick
  if (ticks > max-ticks) or ((pupi-best-tick > 0) ) [stop]
end

;; find leader pigeon and update its fitness value
to find-leader
  ;; leader is best pigeon either follower or walker
  ask min-one-of (turtle-set followers walkers) [value][
    set pupi-leader self     ; update leader
    set pupi-leader-fitness value
    if pupi-leader-fitness < [value] of pupi-best-patch [
        set pupi-best-patch patch-here
        if pupi-best-patch = true-best-patch [ set pupi-best-tick ticks ]
    ]
  ]
end

;; move followers towards pigeon leader
to follow-move
  face pupi-leader fd (distance pupi-leader) * alpha
  rt one-of [0 90 180 270] fd random-normal 0 2  ; a small route deviation due to collisions or wind
  set color blue
end

;; move walkers around
to walk-move
  rt one-of [0 90 180 270] fd (sigma * random-normal 0 1)
  set color green
end

;; turn on the spotlight on the chosen agent
to update-spotlight
  ifelse spotlight = "Pupi best ever"
  [ watch pupi-best-patch]
  [ ifelse spotlight = "True best"
    [  watch true-best-patch ]
    [ ifelse spotlight = "Pupi leader"
      [  watch pupi-leader ]
      [  reset-perspective ]
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
244
10
663
430
-1
-1
2.045
1
10
1
1
1
0
1
1
1
-100
100
-100
100
0
0
1
ticks
30.0

CHOOSER
5
10
110
55
landscape
landscape
"Sphere" "Sphere-offset" "Rastrigin" "Rosenbrock" "Himmelblau" "Eggholder" "Random"
4

BUTTON
5
191
76
224
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
81
191
152
224
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
109
111
142
pop-size
pop-size
0
40
20.0
5
1
NIL
HORIZONTAL

CHOOSER
117
10
228
55
spotlight
spotlight
"Off" "Pupi best" "Pupi leader" "True best"
0

MONITOR
6
282
231
327
pupi best ever
word \"f ( \" \nprecision [x] of pupi-best-patch 2 \", \" \nprecision [y] of pupi-best-patch 2 \" ) = \"\nprecision [value] of pupi-best-patch 6
5
1
11

MONITOR
6
232
230
277
true solution
word \"f ( \" \nprecision [x] of true-best-patch 2 \", \" \nprecision [y] of true-best-patch 2 \" ) = \"\nprecision [value] of true-best-patch 6
5
1
11

SLIDER
117
110
230
143
walkers-rate
walkers-rate
0
1
0.25
0.05
1
NIL
HORIZONTAL

SLIDER
5
148
112
181
alpha
alpha
0
2
0.5
0.1
1
NIL
HORIZONTAL

SLIDER
117
148
230
181
sigma
sigma
0.1
10
2.0
0.1
1
NIL
HORIZONTAL

PLOT
680
14
1019
134
pupi leader fitness
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -955883 true "" "plot pupi-leader-fitness"

BUTTON
158
191
229
224
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
6
332
232
377
pupi current leader
word \"f ( \" \nprecision [x] of pupi-leader 2 \", \" \nprecision [y] of pupi-leader 2 \" ) = \"\nprecision [value] of pupi-leader 6
17
1
11

MONITOR
7
383
113
428
runtime (ms)
ifelse-value (true-best-patch = pupi-best-patch) \n[ (word precision pupi-runtime 3 \"!!!\") ] [pupi-runtime]
2
1
11

PLOT
681
139
1020
259
pupi best solution
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot [value] of pupi-best-patch"

PLOT
681
309
1024
429
pupi cohesion
NIL
NIL
0.0
10.0
0.0
2.0
true
false
"" ""
PENS
"default" 1.0 0 -5825686 true "" "plot pupi-cohesion"

SWITCH
801
272
908
305
cohesion?
cohesion?
0
1
-1000

SLIDER
117
64
229
97
max-ticks
max-ticks
1000
10000
10000.0
1000
1
NIL
HORIZONTAL

MONITOR
120
383
231
428
best tick
ifelse-value (true-best-patch = pupi-best-patch) \n[ (word pupi-best-tick  \"!!!\") ] [pupi-best-tick]
17
1
11

CHOOSER
5
60
110
105
xy-bounds
xy-bounds
6 512
0

@#$#@#$#@
## WHAT IS IT?

This is a reworking as an agent-based model of a recently introduced swarm-based search algorithm for numeric real-valued unconstraint optimisation, inspired in the foraging behaviour of urban pigeons (see [1] for details). The model is intended to find valuable areas or spots in a simulated optimisation landscape, by mimicking how pigeons manage to discover sources of food as they navigate their natural territories, i.e urban parks. The landscape obeys to the variation of a cost function evaluated in the different coordinates of the search space (here, a 2D space). 

We define three distinct pigeon roles or agent types: a leader, who is the pigeon located at the richest source of food at any moment during the simulation, the followers, who are pigeons pursuing the leader in the hope of getting a share of his food, and the walkers, who are pigeons wondering around aimlessly but with an eye looking for food too. 

## HOW IT WORKS

The problem is determined by the LANDSCAPE that is obtained as a discrete projection of the cost function onto the 2D grid of cells comprising the simulation view area. Therefore, pigeons will "search for food" in the projection of said landscape into the view area. Since the purpose of the model is optimisation, the goal is to discover the spot which optimises the value of the cost function evaluated at the coordinates of each cell. Notice that in the current version the algorithm minimises, that is, it searches for a spot with the lowest cost function value. 

Each pigeon is characterised by a location in the landscape (x, y) and the perceived density of food (or cost) in such location which determines its fitness to solve the problem. 

Besides, all pigeons in the population can sense who is the leader (that is, we enable a global information-sharing mechanism). Followers will move towards the leader, so their location is updated in the direction of the leader's location. The walkers, in contrast, move randomly in any direction. Pigeons may change their roles during their lifetime as the simulation progresses, depending on their current fitness.

We defined two types of pigeon breeds, namely followers and walkers. The initial population is created with an amount of pigeons given by the parameter POP-SIZE, with the subset of walkers assigned randomly in proportion to the parameter WALKERS-RATE; the remainder pigeons are assigned to the subset of followers.

Afterwards, each step of the simulation performs four simple actions: find the leader, move the followers, move the walkers and update the best solution found so far. These actions correspond to the following routines: FIND-LEADER (chooses as leader pigeon the one having the best fitness and updates the best fitness ever if necessary), FOLLOW-MOVE (moves each follower towards the leader with the step-size ALPHA, plus a random shift in its orientation due to wind or collisions), and WALK-MOVE (moves each walker around randomly with a step-size SIGMA). These two movement rules correspond to the exploration/exploitation mechanisms of the search algorithm (see [1] for more details).

The simulation is terminated either after a maximum number of steps, MAX-STEPS, or when the truth optimal solution is found prematurely.

## HOW TO USE IT

Firstly choose an optimisation problem to be solved from the LANDSCAPE pull-down list. For any of these problems, then define the appropriate limits of the search space coordinates, namely the XY-BOUNDS. Additionally, choose the algorithm parameters POP-SIZE, WALKERS-RATE, ALPHA and SIGMA. You can also set the termination criterium MAX-TICKS. Then press SETUP, then GO.

The initial location of the population of pigeons will be assigned randomly within the boundaries of the landscape. Afterwards, at each time step pigeons move according to its role, the population fitness is updated, and if needed, the leader is re-assigned. The emergent behaviour of the pigeon flock will show up, while they attempt to discover the promising regions within the landscape; the simulation will show the three breeds of pigeons, leader, followers and walkers with different colors (red, blue and red, respectively).

The output monitors show the location and cost of the true solution for the problem, the best location and best cost ever found by the algorithm during the simulation, and the location and cost associated to the current leader. If the algorithm is able to find the true solution, then the  BEST-TICK and RUNTIME monitors will display a "!!!" sign inserted behind their actual values.

Lastly, the model also outputs the plot of the leader fitness vs time, the plot of fitness of the best solution found vs time and the plot of flock cohesion vs time if the COHESION? switch is enabled. The latter implies an additional cost to the running time, as the model needs to compute distances between all the pigeons in the follower's flock.


## THINGS TO NOTICE

You can see that the flock of follower pigeons moves out from one local minima to another. This is explained because every certain number of ticks, the entire population become walkers that start looking around for other regions as sources of food. This phenomenon is attested by the fitness variation of the leader pigeon during the simulation timeline, as it can be seen in the corresponding plot. Nonetheless, the best found ever solution always has a decreasing fitness as it can be verified in its respective plot.

Similarly, the transition of the flock from one local minimum location to another is depicted in the periodic patterns that appear in the cohesion plot.  

## THINGS TO TRY

The model includes the mathematical expression and projection of a set of widely-known  benchmark functions for unconstrained continuous optimisation: SPHERE, RASTRIGIN, ROSENBROCK, HIMMELBLAU, EGGHOLDER (the actual mathematical expressions and 2D landscapes can be seen in the Appendix of [1]) . In addition to these benchmarks, we defined a RANDOM landscape that is generated on- the-fly with values sampled from a scaled normal distribution; hence, in this problem the real optimum is not known in advance, in contrast to the other functions. Lastly, a seventh benchmark was included, a modified SPHERE with the optimum shifted to the second quadrant. Each benchmark problem exhibits different properties (multi-modality, convexity, separability, etc.) and search ranges (we suggest using a XY-BOUNDS of 512 for EGGHOLDER and XY-BOUNDS of 6 for the other problems).

Notice that all the problems produce a constant landscape (except RANDOM), so you can try and see the effect of varying the different parameters. For starters, a typical configuration can be: POP-SIZE=20, WALKERS-RATE=0.25, ALPHA=0.1, SIGMA=1, MAX-TICKS=10000, XY-BOUNDS=6 (or 512 if LANDSCAPE is EGGHOLDER). If you want to highlight the location of the true solution or the current leader turn on the SPOTLIGHT.  
The RANDOM problem produces a different landscape and true solution each time you press SETUP. It is interesting to see how the pigeon-inspired algorithm is able to solve it nonetheless most of the times.


## EXTENDING THE MODEL

An interesting question arising is if the convergence speed of the algorithm can be improved without compromising its simplicity for practical purposes, for example using time-decay updates of the step sizes of pigeon movements. In addition, the experiments with the RANDOM benchmark hints at the possibility of the model to solve non-stationary problems, that is, problems were landscape may vary over time, an interesting setting for real-world problems.

Other topics for further research are validating wether the ABM approach to swarm intelligence can be extended or is feasible to address optimisation in higher dimensions, different function domains (continuous, binary, combinatorial) or to incarnate other metaphors originating from the field of collective intelligence. 

## RELATED MODELS

Modeling Commons -> Particle Swarm Optimization (PSO), see [2].

## CREDITS AND REFERENCES

Authors:  

Sergio Rojas-Galeano and Martha Garzon 

Copyright (c) July 2020

email: srojas@udistrital.edu.co

Version 1.16

Licenses:

- The model code is licensed as GNU General Public License (GPLv3) 
  (see https://www.gnu.org/licenses/gpl-3.0.txt)

- This Info Tab document is licensed as CC BY-NC-ND 
  (see https://creativecommons.org/licenses/by-nc-nd/4.0/)

References:

[1] Garzon, M., and Rojas-Galeano, S. (2019, November). An Agent-Based Model of Urban   	    Pigeon Swarm Optimisation. In 2019 IEEE Latin American Conference on Computational        Intelligence (LA-CCI) (pp. 1-6). IEEE.  doi: 10.1109/LA-CCI47412.2019.9036758.
    https://ieeexplore.ieee.org/document/9036758

[2] Stonedahl, F. and Wilensky, U. (2008). NetLogo Particle Swarm Optimization model.         http://ccl.northwestern.edu/netlogo/models/ParticleSwarmOptimization. Center for          Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex         Systems, Northwestern University, Evanston, IL.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="xy-bounds">
      <value value="512"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Eggholder&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-alpha" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
      <value value="10"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-sigma" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
      <value value="10"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-popsize" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="20"/>
      <value value="40"/>
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-walkrate" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Base-line-OAT-maxticks" repetitions="25" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[value] of pupi-best-patch</metric>
    <metric>pupi-best-tick</metric>
    <metric>pupi-cohesion</metric>
    <metric>pso-best-tick</metric>
    <metric>pso-best-val</metric>
    <metric>pso-cohesion</metric>
    <enumeratedValueSet variable="landscape">
      <value value="&quot;Sphere&quot;"/>
      <value value="&quot;Sphere-offset&quot;"/>
      <value value="&quot;Rastrigin&quot;"/>
      <value value="&quot;Rosenbrock&quot;"/>
      <value value="&quot;Himmelblau&quot;"/>
      <value value="&quot;Eggholder&quot;"/>
      <value value="&quot;Random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spotlight">
      <value value="&quot;Off&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-ticks">
      <value value="2000"/>
      <value value="5000"/>
      <value value="10000"/>
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cohesion?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pupi-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pso-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="walkers-rate">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sigma">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="particle-inertia">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-attraction">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-limit">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="personal-attraction">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
