patches-own 
[
  ;Economic Fitness Field:
  
  x_t-1 x_t
  Mb
  
  ; Business Cycle Quantum Game Term:
  
  z
  
  ; Economic Dynamics
  
  equilibrium-price_t-1 equilibrium-price_t ; logarithm of equilibrium price
  
  economic-returns ; logarithmic equilibrium price variation
  
  
    ; Financial Returns:
  
  r_t-1 r_t
  Ma
  
  ; Financial Volatility:
  
  sigma_t-1  sigma_t ; this is an intrinsic time parameter that is it coincides with sqrt(tau_i(t)) see p.145 and (15) for explanation of tau_i(t)
  
  
  ; Shares' Price
  s
  
  indicator
  
  c-red
  c-green
  c-blue
]

globals [ ns portfolio-returns mean-fitness sd-volatility]



to setup
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  
  ask patches [ set x_t random-float 1.000 ]
  ask patches [ set r_t 0.01 * (2 * (random-float 1.000) - 1) ]
  ask patches 
  [ set s 1
    set equilibrium-price_t 1 + random-float 9.00 ]
  ask one-of patches [ set indicator 1 ]
end


to go
  tick
  ask patches 
  [ set x_t-1 x_t
    set r_t-1 r_t
    set sigma_t-1 sigma_t ]
    
  business-fitness-dynamics
  
  economic-dynamics
  
  financial-dynamics
  
  
  set mean-fitness mean [x_t] of patches
  
  set sd-volatility mean [abs(r_t)] of patches ; observed volatility can be measured by absolute returns
  
  do-plot
  
  ask patches [colorscheme]
  
  
end


to business-fitness-dynamics
  ask patches [ set z random-normal 0 1.000 ] ; Gaussian wave packet reduction around the standardized fitness operator
  ask patches [ set Mb (1 - m) * (b * x_t-1 - (b + 1) * x_t-1 ^ 3) + m * r_t-1 ] ; cubic map update (equation (18) with Mb := f_b,m)
  ask patches [ set x_t (1 - epsilon - gamma) * Mb + epsilon * mean [ Mb ] of patches + gamma * z ] ; F update and result of the quantum wave packet reduction in terms of 
                                                                                                    ; the fitness field operator eigenvalue
end


to economic-dynamics
  ask patches [set equilibrium-price_t-1 equilibrium-price_t] ; update of economic equilibrium price logarithms (equations (2) and (3))
  ask patches 
  [set economic-returns theta_0 * (miu - equilibrium-price_t-1) + theta_1 * x_t ; update of logarithmic growth rate in equilibrium price, or "economic equilibrium price returns" 
                                                                                ; as per (equation (3))
   set equilibrium-price_t equilibrium-price_t-1 + economic-returns ] ; new equilibrium price
end



to financial-dynamics
  returns-dynamics
  portfolio-dynamics
end




to returns-dynamics
  ask patches [ set sigma_t phi0 * phi1 + phi0 * sigma_t-1 + (phi1 + sigma_t-1) * x_t ^ 2 ] ; volatility dynamics defined in terms of the financial intrinsic time equation
  ask patches [ set r_t r0 + (1 - epsilon_A) * sigma_t * x_t + epsilon_A * mean [ sigma_t * x_t ] of patches with [abs(x_t - [x_t] of myself) <= delta_A] ]
  ; for the last command line see equation (14) and condition (20), pages: 45 and 147.
end


to portfolio-dynamics
  ; for the portfolio dynamics see equations (24) and (25) in page 151:
  ask patches [ set s s * exp(r_t)]
  set ns sum [ s ] of patches
  set portfolio-returns sum [ (s / ns) * (exp(r_t) - 1) ] of patches
end

to do-plot
  set-current-plot "Mean Field Fitness"
  plot mean-fitness
  set-current-plot "Portfolio Returns"
  plot portfolio-returns
  set-current-plot "Local Returns"
  plot [r_t] of one-of patches with [indicator = 1]
  set-current-plot "Volatility Synchronization"
  plot sd-volatility
  set-current-plot "Phase Space"
  plotxy mean [equilibrium-price_t] of patches mean [ln(s)] of patches
 
end



to colorscheme
  set c-red int ((abs(r_t) * 50) * 255)
  set c-green int ((abs(r_t) * 50) * 255)
  set c-blue int ((abs(r_t) * 50) * 255)
  if c-red > 255 [set c-red 255]
  if c-green > 255 [set c-green 255]
  if c-blue > 255 [set c-blue 255]
  set pcolor (list c-red c-green c-blue )  
end
@#$#@#$#@
GRAPHICS-WINDOW
1056
25
1301
226
-1
-1
17.0
1
10
1
1
1
0
1
1
1
0
9
0
9
0
0
1
ticks
30.0

SLIDER
13
99
105
132
b
b
0
3
3
0.0001
1
NIL
HORIZONTAL

SLIDER
125
100
217
133
m
m
0
1
0.01
0.01
1
NIL
HORIZONTAL

SLIDER
14
155
106
188
epsilon
epsilon
0
1
0.05
0.001
1
NIL
HORIZONTAL

SLIDER
132
154
224
187
gamma
gamma
0
1
0.0010
0.001
1
NIL
HORIZONTAL

SLIDER
18
223
110
256
phi0
phi0
0
1
0.5
0.01
1
NIL
HORIZONTAL

SLIDER
132
224
224
257
phi1
phi1
0
1
0.0010
0.001
1
NIL
HORIZONTAL

SLIDER
131
276
223
309
epsilon_A
epsilon_A
0
1
0.8
0.01
1
NIL
HORIZONTAL

SLIDER
21
276
113
309
r0
r0
0
1
1.0E-5
0.00001
1
NIL
HORIZONTAL

SLIDER
8
337
120
370
delta_A
delta_A
0
1
0.2
0.01
1
NIL
HORIZONTAL

BUTTON
138
10
201
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
222
10
285
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
312
10
687
160
Portfolio Returns
NIL
NIL
0.0
10.0
-0.0010
0.0010
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
308
171
691
321
Local Returns
NIL
NIL
0.0
10.0
-0.01
0.01
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
701
337
1101
506
Mean Field Fitness
NIL
NIL
0.0
10.0
-0.0010
0.0010
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
307
336
696
507
Volatility Synchronization
NIL
NIL
0.0
10.0
0.0
0.0010
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

SLIDER
14
52
106
85
theta_0
theta_0
0
1
0.01
0.01
1
NIL
HORIZONTAL

SLIDER
126
52
218
85
theta_1
theta_1
0
1
0.9
0.1
1
NIL
HORIZONTAL

SLIDER
12
10
104
43
miu
miu
0
100
3.7
0.1
1
NIL
HORIZONTAL

PLOT
711
16
1024
301
Phase Space
<ln[Equilibrium Price (t-1)]>
<ln[Financial Price (t)]>
5.0
6.0
0.0
0.01
true
false
"" ""
PENS
"default" 1.0 2 -16777216 true "" ""

@#$#@#$#@
## THE MODEL

This is an integrated model of a coevolving real economy and financial market, proposed in the article:

"Financial Turbulence, Business Cycles and Intrinsic Time in an Artificial Economy", Gon_alves, C.P.; Algorithmic Finance (2012), 1:2, 141-156.

The model is among one of the first applications of quantum econophysics to combine quantum chaos and quantum optimization to the integrated financial modeling of real economy and financial market coevolving dynamics, this is done within an integrated modeling approach that shows the emergence of turbulence and scaling patterns in the financial returns' dynamics, resulting from the coevolution of competing companies in a market economy and a financial market comprised of value investors and arbitrageurs, thus establishing a bridge to evolutionary arbitrage theory.

Throughout the model review and tutorial, below, we will sometimes mention equation numbers, these correspond to equations in the article, available online at:

http://algorithmicfinance.org/1-2/pp141-156/




## APPROACH TO QUANTUM EVOLUTIONARY FINANCIAL ECONOMICS

The current model is aimed at a modeling gap between artificial financial markets finance-driven modeling and economic nonlinear dynamics' modeling.

Indeed, a model of financial dynamics without the underlying business economic dynamics is incomplete, since the value-driver dynamics is either absent or tucked away to the form of a global noise term, on the other hand, a model of economic dynamics without the financial dynamics is also incomplete, since the financial system is not simply an evaluator it is also a source of business facts and dynamics.

Thus, the quoted article, and the present model move to the next direction in financial and economic modeling: to build a model of an artifical economy in order to address the coevolution between the real economy and the financial system.

Thus, we are dealing with the disciplinary context of evolutionary financial economics, defined as the scientific discipline that concerns itself with the evolutionary processes and dynamics that take place within a systemic framework of coevolution between real economies and financial systems.

A quantum econophysics-based approach to evolutionary financial economics is followed. Econophysics is concerned with applications of statistical and quantum mechanics to economics, the applications of quantum mechanics (including quantum statistical mechanics and quantum field theory) to economics is called quantum econophysics, it is this later framework that is adopted here.

There are three main advantages of the quantum approach to Evolutionary Financial Economics:

- The explanatory effectiveness is expanded by the fact that one does not need any prior probability assumption, instead, one models the system's inter-relations and dynamics and from that result dynamical probabilities.

- Probabilities can have evolutionary and game theoretical interpretations.

- The adaptation process of a Complex Adaptive System (CAS) can be fully integrated with the probability formation and quantum game equilibrium assumptions.

In the article and model, we follow the conceptual basis of Complex Quantum Systems Science and Quantum Chaos applied to Evolutionary Financial Economics, with methodological consequences regarding:

- Combination of quantum lattice field theory with quantum optimization;

- Applications of complex quantum systems models, in particular dissipative quantum chaos, to financial theory.

Besides these two contributions to the field of quantum econophysics, the model links financial modeling to business cycle modeling, making emerge financial turbulence from a coevolving dynamics with the real economy.

## HOW IT WORKS

The model works with a business lattice fitness field described by a field operator X(i) (in the article, it is represented by "x-hat underscore i", i being the lattice site, in this model description we use the notation X(i) due to limitations of text format in placing the hat above the X).

The field operator is assumed to commute between each pair of different lattice sites, thus, [X(i),X(j)] = delta(i,j), and has a continuous spectrum that spans a Rigged Hilbert space at each site (equation (4)). At each lattice site (called "patch" in the 2x2 lattice of the Netlogo world), the field eigenvalues will work as the configuration space for a wave function (it is actually a wave functional over the fitness field).

The quantum game is divided in rounds, with a local harmonic oscillator Hamiltonian condition and optimization problem defined by the minimization of the potential energy.

The potential is fixed for each round, but changes from round to round through a nonlinear map update rule.

The resulting optimization leads to the Gaussian wave packet solution corresponding the harmonic oscillator ground state:

     ps_i,t(x) = [1 / (gamma * sqrt(2*pi))] ^ 0.5 * exp[-u(t)/(4 * gamma ^ 2)]


     u(t) = (x - F(x_i(t-1)) ^ 2

where gamma represents the square root of business cycle intrinsic time. The wave function leads, at the end of each round t, to a certain fitness field value at site i with a gaussian probability distribution.

F is a nonlinear map, whose structure is given by equations (16) to (18), on the other hand, equations (14) and (15) formalize the financial dynamics, which depends upon two types of agent:

- Value investors, that evaluate the local fitness field: aiming at reflecting the fitness value at each site (company);

- Arbitrageurs, that look for companies that are considered to be close in fitness to the company under valuation and, thus, act as a price-to-value convergence force.

Both arbitrageurs and value investor trading takes place with an intrinsic time rescaling, linked to financial valuation of financial volatility risk versus business fitness risk (equation (15)).

The model combines chaotic dynamics, linked to the nonlinear map F, with quantum fluctuations linked to the wave localization on a definite field value at each site.

This allows for previous quantum wave reductions coupled to nonlinear dynamics to affect the Hamiltonian for the next round of the game, at each site.

The combination of the quantum stochastic and chaotic dynamics affects the wave function at each round, through the quantum game optimization problem, such that the quantum state transition for the quantum game equilibrium solutions psi_i,t-1(x) -> psi_i,t(x) can be accounted for by a round-to-round connector in the form of a quantum Radon-Nikodym operator, that is, we have, for the quantum game equilibrium dynamics:


     |psi_i,t> = K(t,t-1) |psi_i,t-1>

where K(t,t-1) is an operator satisfying:

     K(t,t-1)|x> = K(x,t,t-1)|x>


     K(x,t,t-1) = exp{ (2*x*[F(x_i(t-1)) - F(x_i(t-2))] - [F(x_i(t-1))^2 - F(x_i(t-2))^2]) /(4 * gamma ^ 2) }


from where it follows that:

     psi_i,t(x) = K(x,t,t-1) * psi_i,t-1(x)


We are dealing, in this case, with a conditional wave functional, conditional upon the fitness value at the beginning of the round (x_i(t-1)). The wave functional at the end of each round could either be obtained by a conditional Schr_dinger wave equation with a rotated solution, so that at the end of each round the wave functional would coincide with the eigenfunction of the Hamiltonian problem without any chronological time-dependent phase of the form exp(iE(t-t') / h-bar_s), or, alternatively, we can work with the round-dependent Hamiltonian problem plus quantum optimization and connect the two round's quantum game equilibrium solutions by a quantum Radon-Nikodym operator.


It is called Radon-Nikodym operator because the effect of this operator, applying Born's rule to calculate the probability distribution, is similar to a Radon-Nikodym derivative applied to connect two probability distributions (in this case two Gaussian densities).

NOTE: in quantum computation one can define the transition between two superposition states in a diagonalized form on the "logical" basis in terms of quantum Radon-Nikodym operators, this can be easily shown for a single qubit, for instance, in the transition:

     |psi> = psi(0)|0> + psi(1)|1> -> |phi> = phi(0)|0> + phi(1)|1>

As long as psi(0) and psi(1) are both different from zero we can write:

     K(|phi>,|psi>)|s> = (phi(s)/psi(s))|s>, s=0,1

from where it follows that: K(phi,psi)|psi> = |phi>. The Radon-Nikodym operator cannot, however, produce the following entanglement operation:

	psi(0)|00> + psi(1)|10> -> psi(0)|00> + psi(1)|11>

This is because the initial state is a pointer state. On the other hand, entanglement can also take place in a case of a unitary transition that has an "equivalent" Radon-Nikodym operator, for instance, consider the following 2-qubit state:


	|psi,phi> = phi(0)(psi(0)|00> + psi(1)|10>) + phi(1)(psi(0)|01> + psi(1) |11>)

One can obtain entanglement through a sequence of unitary quantum gates with the output state:

	|psi> = psi(0) |00> + psi(1)|11>


In this case, there is also a corresponding quantum Radon-Nikodym operator, that can be defined as:

	K(|psi,phi>,|psi>)|ss> = (1 / phi(s))|ss> 
	K(|psi,phi>,|psi>)|s1-s> = 0|s1-s>

Applying this operator to |psi,phi> would yield the entangled state  psi(0) |00> + psi(1)|11>.

One can check the above equivalence with a straightforward example of the initial state |+,+> = |+>|+>, in this case, the unitary transition is given by an inverse Haddamard transform on the second qubit, which would yield |+,0> followed by a C-NOT gate that would yield the desired entangled state |psi> = (1/sqrt(2))(|00> + |11>), the corresponding Radon-Nikodym operator would be:

	K(|+,+>,|psi>)|ss> = sqrt(2)|ss> 
	K(|+,+>,|psi>)|s1-s> = 0|s1-s>

This example allows us to illustrate that only some unitary transitions can be addressed by an equivalent quantum Radon-Nikodym operator.

In the current work, the quantum Radon-Nikodym operator is assumed in the formalism as it allows for a useful link between standard financial mathematics (which works with Radon-Nikodym derivatives) and it is effective for conditional quantum optimization problems leading to wave function trajectories incorporating end-of-round wave packet reduction and chaotic dynamics in a path-dependent quantum adaptive computation framework.

## HOW TO USE IT

The interface is divided in a number of sliders that allow the user to control both the economic and financial parameters, as is now explained.

A) ECONOMIC PARAMETERS:


miu, theta_0 and theta_1 are the three main parameters of equation (3), for the mean reversing dynamics of the logarithmic growth rate in the economic equilibrium price (see Eqs.(1) and (2) for the equilibrium and growth definition).

b corresponds to the cubic map's parameter, as per equation (18), while m is the coupling parameter between the business fitness dynamics and the financial returns (as per equation (17)), introducing an impact on business dynamics coming from the financial system (for further details see the "MASS DISCUSSION AND EXPERIMENTS" section below).


epsilon is a global coupling constant for a mean field coupling between companies in the fitness dynamics (as per equation (16)), accounting for the competition between firms. 

gamma corresponds to the square root of a business cycle intrinsic time scale and defines the level of economic volatility (for further details see the "MASS DISCUSSION AND EXPERIMENTS" section below).


B) FINANCIAL PARAMETERS:

phi0 and phi1 correspond to the two main parameters for the financial intrinsic time dynamics, as per equation (15).

r0 is a fixed component in the financial returns.

epsilon_A and delta_A are, respectively, the proportion of arbitrageurs in the financial system and the arbitrage threshold.

There are three influences upon the financial returns, in accordance with equation (14):

- The fixed average returns component (r0);

- The value investors component, which has an impact (or coupling) of (1 - epsilon_A)

- The arbitrageurs component, which has an impact (or coupling) of epsilon_A


Arbitrage leads to a dynamic networking of connections, in the sense that the arbitrage coupling for each company is not fixed, arbitrageurs try to identify which companies have value fitness in a close neighborhood of each other (of no more than delta_A, in terms of Euclidian distance) and, thus, trade in such a way that they take advantage of the mismatching of returns, leading to a returns' convergence that is modeled in terms of a mean taken over the arbitrage cluster of each company, this mean is evaluated for the companies that comprise the cluster and is the mean of the product of the square root of each company's financial intrinsic time by the respective fitness.

The main assumption is of a business fitness-seeking trading dynamics, so that returns with similar fitness should be closely matched, the arbitrage coupling is, thus, coevolving with the economic and financial dynamics.

The value investors, on the other hand, impact on financial returns by evaluating each company's fitness multiplied by the respective financial intrinsic time, keeping in line with the above assumption of business fitness-seeking trading dynamics.


Besides the parameters, the model's interface shows six outputs to the user:

- A market portfolio returns plot;

- The returns for a single company, chosen at random at the beginning of the simulation;

- The mean field of business fitness;

- The "Phase Space" plot, defined in terms of the companies' mean (economic) equilibrium price logarithm versus the companies' mean shares price logarithm at each round;

- The volatility synchronization plot, defined in terms of the mean absolute returns of the population of companies (this is a market-observable volatility, which can be calculated for actual financial data);

- The Netlogo world, itself, which makes correspond the absolute returns of each patch (company) to a greyscale color scheme, so that lighter colors correspond to stronger volatility, while darker colors correspond to lower volatility, as measured by the absolute returns.



## MASS DISCUSSIONS AND EXPERIMENTS


As a first point, some of the parameters that we will be addressing in this section were not explicitly placed in the model's sliders, but they underlie some of the parameters that are present in the sliders. As we move along this section, we will address the sliders. Having made this first note, we now address the mass problem and experiments that the user can try out.


A) THE MASS ISSUE

In this market model, the mass (or the mass-like term) enters first in the Hamiltonian problem. Indeed, given the potential function for the i-th company at round t:


	V(x) = V(x,i,t) = (a/2) * ( x - F(x_i(t-1)) ) ^ 2


The parameter "a" represents the business evolutionary pressure, such that the higher the parameter a's value is the more competitive is the business environment, as explained in the article.

Now, given the harmonic oscillator's mass "m" and "a" we obtain the oscillation angular frequency measured in radians:

	omega = (a / m) ^ 0.5

from this last relation comes:

	m = a * omega ^ -2

Since, in the economic setting, "a" is dimensionless, the business cycle mass-like term is expressed in units of inverse squared angular frequency and is assumed as characteristic of the industry itself.

Thus, the higher the mass is, the lower is the oscillation frequency, and vice-versa, so that faster-paced economic rhythms or slower-paced economic rhythms are being addressed by this relation between mass and angular frequency.

On the other hand, the parameter gamma is also related to the economic mass-like term, indeed, given that gamma = sqrt(tau) and tau is defined as the mean potential energy, calculated under the round's wave function, multiplied by 2 / a, then, it follows that:

	tau = (h-bar_s * omega) / 2a = (h-bar_s / 2) * (a * m) ^ -0.5

Thus, the higher the mass is, the slower is the economic intrinsic time, such intrinsic time is not measured in clock time, rather it is interpreted in terms of economic rhythms that rescale volatility with the usual square root rule that holds for clock-based temporal intervals.

This intrinsic time scale, simultaneously, relates the three main constants of the economic game: h-bar_s, a and m.

In the model's sliders we omitted "a" and "h-bar_s" opting to work with "gamma" and "m".

Now, the user can perform different experiments regarding the massive coupling between the financial system and the economic system.

The first two experiments work with a fully decoupled setting. To begin the first experiment set the parameters as follows:

- miu = 3.7, theta_0 = 0.01, theta_1 = 0.9, b = 2.84, epsilon = 0, epsilon_A = 0, r = 1.0E-5, phi0 = 0.5, phi1 = 0.001, gamma = 0.003 and m = 0

In this case, we have set m = 0, just like in Fig.1 of the article, which means that we are considering a decoupling of the economic dynamics with respect to the financial dynamics. We must be careful, however, in interpreting this value for m, what is meant by m = 0, both in the article and in this experiment, is that we are setting the mass coupling in equation (17) to zero, of course, the mass "m" still enters the economic dynamics through gamma, it just does not enter into the equation for the coupling of the economic dynamics to the financial dynamics, so it is set to zero in that equation specifically.

Run the model for a while and watch closely the resulting dynamics, then, change the parameter gamma reducing it by -0.001 running again the model until reaching 0.

What you have done by reducing gamma is to zero in on the periodic window shown in figure 1 of the current article.

The basic meaning of the procedure that you just undertook is the following:


Allowing for finite and fixed "a * m" component, reducing h-bar_s, transforms the Gaussian wave packets that characterize each round to increasingly tight distributions, until, in the limit, the |psi_i,t(x)|^2 collapses to a Dirac distribution centered at F(x_i(t-1)), such that the system follows a classical trajectory, this means that, while the economic dynamics stabilizes in a periodic dynamics, in accordance with the classical periodic window (just like in Fig.1 bifurcation map), the financial dynamics suffers an oscillating explosive process, which busts the market in terms of volatility and produces an oscillating exponential growth in the portfolio returns.

The dynamics is, in this case, unsustainable.

The birth of the periodic window in the classical dynamics for h-bar_s -> 0, does not have a corresponding quantum dynamics for h-bar_s sufficiently large, where the quantum fluctuations lead to dynamical chaos "destroying" the classical periodic window.

Let us, then, allow for gamma = 0.001, running the model with this parameter produces the explosive pattern, but if we let m be equal to 0.01 and, then, run the dynamics again, the result is quite different. The massive interaction between financial system and the economic system's dynamics introduces sufficient feedback to break the oscillating explosive dynamics, however, the resulting dynamics does not show evidence of the usual financial turbulence pattern.

Raising b to 3 allows you to recover the usual turbulent pattern at the microscopic (i.e., company) level.

The portfolio, on the other hand, for this new parameter setting, smooths out the turbulence, this no longer holds, however, if the spatial coupling is increased. Even for epsilon = 0.05, implying a high level of differentiation between the companies, the increased coupling is enough to lead to turbulence at the portfolio returns level.

Increased coupling will change the economic and financial dynamics qualitatively, depending upon the coupling level.

Considering again a lower epsilon = 0.01, where the portfolio is still effective in reducing turtulence for the investor, and epsilon = 0.05, one may try to address the effects of arbitrage in each case.

As it turns out, for epsilon = 0.01 with the arbitrage coupling set to 0.1, and delta_A = 0.22, we see that the portfolio returns will start to show evidence of turbulence and higher volatility risk, including the occurrence of jumps. For epsilon = 0.05, we can see that arbitrage in no way removes the turbulence from the portfolio returns.

Thus, in this model, arbitrage does not necessarily lead to a lowering of financial turbulence, it can have the opposite effect.


## ADDED MATERIAL

At http://youtu.be/IXBskXwr-c8 is shown a video with an added experiment set and discussion on arbitrage and the coevolving financial and economic dynamics.



## REFERENCES

"Financial Turbulence, Business Cycles and Intrinsic Time in an Artificial Economy", Gon_alves, C.P.; Algorithmic Finance (2012), 1:2, 141-156.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
